#region Copyright (c) Aquineo
/*
 * License
 * 
 * Aquineo disclaims all warranties with regard to this software, including all 
 * implied warranties of merchantability and fitness, in no event shall Aquineo 
 * be liable for any special, indirect or consequential damages or any damages 
 * whatsoever resulting from loss of use, data or profits, whether in an action 
 * of contract, negligence or other tortious action, arising out of or in 
 * connection with the use or performance of this software. 
 * 
 * Permission to use, copy, modify, and distribute this software for any purpose 
 * and without fee is hereby granted, subject to the following restrictions: 
 * 
 * 1. The origin of this software must not be misrepresented; you must not claim 
 * that you wrote the original software. If you use this software in a product, 
 * an acknowledgment (see the following) in the product documentation and this 
 * permission notice is required:
 *
 * Copyright  2004 Aquineo (www.aquineo.com)
 * 
 * 2. Altered source versions must be plainly marked as such, and must not be 
 * misrepresented as being the original software.
 * 
 * 3. This notice may not be removed or altered from any source distribution.
 * 
 * */
#endregion

/*
 * Classes: CIni
 *
 * History: 
 *	2004-02-02, created
 * 
 * Remarks:
 *
 * ToDo:
 * */

using System;
using System.Collections;
using System.IO;
using System.Text;

namespace Planet.Utility.Ini
{

	/// <summary>This class manages an INI file.</summary>
	/// <remarks>For a detailed description of the Aquineo INI file
	/// format see the documentation (Ini.html).</remarks>
	public class CIni
	{
		#region constructors
		/// <summary>Constructor</summary>
		/// <param name="FileName">Path and name of INI file.</param>
		public CIni(string FileName)
		{
			try 
			{
				this.FileName = FileName;
				Read(this.FileName);
			}
			catch { throw; }
		}

		/// <summary>Constructor</summary>
		/// <param name="FileName">Path and name of INI file.</param>
		/// <param name="TextEncoding">Text encoding of the INI file.</param>
		public CIni(string FileName, Encoding TextEncoding)
		{
			try
			{
				this.FileName = FileName;
				this._TextEncoding = TextEncoding;
				Read(this.FileName);
			}
			catch { throw; }
		}
		#endregion

		#region constants
		/// <summary>Enumeration of possible line terminators.</summary>
		public enum LINE_TERMINATOR : uint
		{
			/// <summary>Carriage return followed by line feed.</summary>
			CR_LF = 1,
			/// <summary>Line feed followed by carriage return.</summary>
			LF_CR = 2,
			/// <summary>Carriage return.</summary>
			CR = 3,
			/// <summary>Line feed.</summary>
			LF = 4
		}

		/// <summary>Parser position during parsing of INI file.
		/// </summary>
		protected enum PARSE_POS : uint
		{
			/// <summary>This range covers comment lines before a section
			/// name line and the line with the section's name itself.</summary>
			A = 1,
			/// <summary>This range covers the lines of a section after
			/// the section name line towards the ending of the section.
			/// </summary>
			B = 2
		}

		/// <summary>Introduces a comment.</summary>
		public const char COMMENT_DELIMITER = ';';
		/// <summary>Characters not allowed inside of a comment.</summary>
		public static readonly char[] INVALID_COMMENT_CHARS =
			new char[]{'\n', '\r'};
		#endregion

		#region members and properties
		/// <summary>See <see cref="TextEncoding"/>.</summary>
		protected Encoding _TextEncoding = Encoding.UTF8;
		/// <summary>Text encoding of the INI file.</summary>
		public Encoding TextEncoding
		{
			get
			{
				return _TextEncoding;
			}
		}

		/// <summary>See <see cref="FileName"/>.</summary>
		protected string _FileName;
		/// <summary>Path and name of the attached INI file.</summary>
		public string FileName
		{
			get 
			{
				return _FileName;
			}
			set
			{
				_FileName = value.Trim();
			} // set
		}

		/// <summary>Array of sections, see <see cref="CIniSectionItem"/>.</summary>
		protected ArrayList Sections = new ArrayList();

		/// <summary>Specifies the line terminator. The default line
		/// terminator is <see cref="CIni.LINE_TERMINATOR.CR_LF"/>.</summary>
		public static LINE_TERMINATOR LineTerminator = LINE_TERMINATOR.CR_LF;
		#endregion

		#region Escape(string)
		/// <summary>Encodes double quotation marks as escape sequences, 
		/// containing a backslash followed by a double quotation mark.</summary>
		/// <param name="Value">Input string.</param>
		/// <returns>Encoded string.</returns>
		public static string Escape(string Value)
		{
			try
			{
				if (Value == null)
				{
					return null;
				}
				else
				{
					return Value.Replace("\"", "\\\"");
				}
			}
			catch { throw; }
		}  
		#endregion

		#region Unescape(string)
		/// <summary>Decodes escape sequences, containing a backslash followed 
		/// by a double quotation mark, as a double quotation mark.</summary>
		/// <param name="Value">Input string.</param>
		/// <returns>Decoded string.</returns>
		public static string Unescape(string Value)
		{
			try
			{
				if (Value == null)
				{
					return null;
				}
				else
				{
					return Value.Replace("\\\"", "\"");
				}
			}
			catch { throw; }
		}
		#endregion

		#region GetLineTerminator()
		/// <summary>Gets the line terminator string, see
		/// <see cref="LineTerminator"/>.</summary>
		/// <returns>Line terminator string.</returns>
		public static string GetLineTerminator()
		{
			try
			{
				string ReturnVal = null;

				switch (LineTerminator)
				{
					case LINE_TERMINATOR.CR_LF:
						ReturnVal = "\r\n";
						break;
					case LINE_TERMINATOR.LF_CR:
						ReturnVal = "\n\r";
						break;
					case LINE_TERMINATOR.CR:
						ReturnVal = "\r";
						break;
					case LINE_TERMINATOR.LF:
						ReturnVal = "\n";
						break;
				} // switch

				return ReturnVal;
			}
			catch { throw; }
		}
		#endregion

		#region IndexOfSection(string)
		/// <summary>Searches for the first occurrence of a section.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <returns>The zero-based index of the first occurrence of 
		/// a section within the entire ArrayList <see cref="Sections"/>, 
		/// if found; otherwise, -1.</returns>
		public int IndexOfSection(string SectionName)
		{
			try
			{
				int Index = -1;
				for (int i = 0; i < Sections.Count; ++i)
				{
					if (((CIniSectionItem)Sections[i]).Name == SectionName)
					{
						Index = i;
						break;
					}
				}
				return Index;
			}
			catch { throw; }
		}
		#endregion

		#region IsSection(string)
		/// <summary>Checks existence of a section.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <returns>True if section exists.</returns>
		public bool IsSection(string SectionName)
		{
			try
			{
				if (!CIniSectionItem.IsValidSectionName(SectionName)) 
				{ 
					return false; 
				}
				else
				{
					int Index = IndexOfSection(SectionName);
					if (Index > -1 && Sections[Index] != null)
					{
						return true;
					}
					else
					{
						return false;
					}
				} // else
			}
			catch { throw; }
		}
		#endregion

		#region IsKey(string, string)
		/// <summary>Checks existence of a key inside of a section.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>True if key exists.</returns>
		public bool IsKey(string SectionName, string KeyName)
		{
			try
			{
				if (!CIniSectionItem.IsValidSectionName(SectionName) ||
					  !CIniKeyItem.IsValidKeyName(KeyName))
				{
					return false;
				}
				else
				{
					int Index = IndexOfSection(SectionName);
					if (Index > -1 && Sections[Index] != null && 
						  ((CIniSectionItem)Sections[Index]).GetKey(KeyName) != null)
					{
						return true;
					}
					else
					{
						return false;
					}
				} // else
			}
			catch { throw; }
		}
		#endregion

		#region ToString()
		/// <summary>Returns the entire INI file.</summary>
		/// <returns>INI file as a string.</returns>
		public override string ToString()
		{
			try
			{
				StringBuilder IniItem = new StringBuilder(String.Empty);

				if (Sections != null && Sections.Count > 0)
				{
					for (int i = 0; i < Sections.Count - 1; ++i)
					{
						IniItem.Append(((CIniSectionItem)Sections[i]).ToString());
						IniItem.Append(GetLineTerminator());
					} // for
					IniItem.Append(
						((CIniSectionItem)Sections[Sections.Count - 1]).ToString());
				} // if

				return IniItem.ToString();
			}
			catch { throw; }
		}
		#endregion

		#region AddSection(string, ArrayList)
		/// <summary>Adds a section and its optional comment lines. Overrides
		/// existing sections with the same name.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <param name="LeadingComments">Comment lines, must be null if 
		/// comment is missing.</param>
		/// <returns>Returns a reference to the added section element.</returns>
		/// <exception cref="ArgumentException">Invalid section name,
		/// see <see cref="CIniSectionItem.IsValidSectionName(string)"/>.
		/// </exception>		
		public CIniSectionItem AddSection(string SectionName, 
			ArrayList LeadingComments)
		{
			try
			{
				if (!CIniSectionItem.IsValidSectionName(SectionName))
				{
					throw new ArgumentException();
				}

				// Override existing section.
				int Index = IndexOfSection(SectionName);
				if (Index > -1)
				{
					// Section exists, override section.
					Sections[Index] = new CIniSectionItem(SectionName, LeadingComments);
				}
				else
				{
					// Create new section.
					Index = Sections.Add(new CIniSectionItem(SectionName, LeadingComments));
				}
				
				return (CIniSectionItem)Sections[Index];
			}
			catch { throw; } 
		} 
		#endregion

		#region AddSection(CIniSectionItem)
		/// <summary>Adds a section element, overrides existing sections with the
		/// same name.</summary>
		/// <param name="SectionItem">Section element to be added.</param>
		/// <returns>Returns a reference to the added section element.</returns>	
		/// <exception cref="ArgumentException">Invalid section name,
		/// see <see cref="CIniSectionItem.IsValidSectionName(string)"/>.
		/// </exception>
		/// <exception cref="ArgumentNullException">Null as an argument.</exception>
		public CIniSectionItem AddSection(CIniSectionItem SectionItem)
		{
			try
			{
				if (SectionItem == null)
				{
					throw new ArgumentNullException();
				}
				if (!CIniSectionItem.IsValidSectionName(SectionItem.Name))
				{
					throw new ArgumentException();
				}

				// Override existing section.
				int Index = IndexOfSection(SectionItem.Name);
				if (Index > -1)
				{
					// Section exists, override section.
					Sections[Index] = SectionItem;
				}
				else
				{
					// Create new section.
					Index = Sections.Add(SectionItem);
				}
				
				return (CIniSectionItem)Sections[Index];
			}
			catch { throw; } 
		} 
		#endregion

		#region GetSection(string)
		/// <summary>Search for the first occurrence of a section.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <returns>Section, if found; otherwise, null.</returns>
		public CIniSectionItem GetSection(string SectionName)
		{
			try
			{
				int Index = IndexOfSection(SectionName);
				if (Index > -1)
				{
					// Element exists, value is not null.
					return (CIniSectionItem)Sections[Index];
				}
				else
				{
					// Element doesn't exist.
					return null;
				}
			}
			catch { throw; }
		}
		#endregion

		#region RemoveSection(string)
		/// <summary>Deletes a section. If the name of the section is valid but 
		/// doesn't exist no operation is performed.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <exception cref="ArgumentException">Invalid section name,
		/// see <see cref="CIniSectionItem.IsValidSectionName(string)"/>.
		/// </exception>
		public void RemoveSection(string SectionName)
		{
			try
			{
				if (!CIniSectionItem.IsValidSectionName(SectionName))
				{
					throw new ArgumentException();
				}

				if (Sections.Count > 0)
				{
					int Index = IndexOfSection(SectionName);
					if (Index > -1) { Sections.RemoveAt(Index); }
				}
			}
			catch { throw; }
		}
		#endregion

		#region CountValues(string, string)
		/// <summary>Gets the number of values of a corresponding section 
		/// and key.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>Number of values.</returns>
		/// <exception cref="NullReferenceException">Section or key
		/// doesn't exist.</exception>
		/// <remarks>To standardise behaviour this method must throw an 
		/// exception if the section or key doesn't exist (<see 
		/// cref="CIniKeyItem.CountValues"/> always returns a number 
		/// greater -1).</remarks>
		public int CountValues(string SectionName, string KeyName)
		{
			try
			{
				if (IsKey(SectionName, KeyName))
				{
					return GetSection(SectionName).GetKey(KeyName).CountValues;
				}
				else
				{
					throw new NullReferenceException();
				}
			}
			catch { throw; }
		}
		#endregion

		#region GetValue(string, string, int, bool)
		/// <summary>Gets a value of a corresponding section and key.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <param name="KeyName">Name of a key.</param>
		/// <param name="IndexOfValue">The zero-based index of a value. Maximum
		/// index is <see cref="CountValues(string, string)"/> - 1.</param>
		/// <param name="IgnoreQuoteFlag">True will return the sole string value
		/// not enclosed in <see cref="CIniValueItem.QUOTE_DELIMITER"/>. 
		/// False encloses the value in <see cref="CIniValueItem.QUOTE_DELIMITER"/>
		/// if the quote flag is set.</param>
		/// <returns>String value, if found; otherwise, null.</returns>
		/// <exception cref="ArgumentOutOfRangeException">IndexOfValue is
		/// out of range.</exception>
		public string GetValue(string SectionName, string KeyName, 
			int IndexOfValue, bool IgnoreQuoteFlag)
		{
			try
			{
				if (IsKey(SectionName, KeyName))
				{
					return GetSection(SectionName).GetKey(KeyName).
						GetValue(IndexOfValue, IgnoreQuoteFlag);
				} 
				else
				{
					return null;
				}
			}
			catch { throw; }
		}
		#endregion

		#region SetValue(string, string, string, bool)
		/// <summary>Sets the value of a corresponding section and key
		/// after clearing all existing values. Both section and key 
		/// will be created, if not found.</summary>
		/// <param name="SectionName">Name of a section.</param>
		/// <param name="KeyName">Name of a key.</param>
		/// <param name="Value">Value.</param>
		/// <param name="IsQuoted">Specifies quotation of a value, see 
		/// <see cref="CIniValueItem.QUOTE_DELIMITER"/>.</param>
		public void SetValue(string SectionName, string KeyName,
			string Value, bool IsQuoted)
		{
			try
			{
				if (!IsSection(SectionName)) 
				{ 
					AddSection(SectionName, null); 
				}
				if (!IsKey(SectionName, KeyName))
				{
					GetSection(SectionName).AddKey(KeyName, null);
				}

				GetSection(SectionName).GetKey(KeyName).ClearValues();
				GetSection(SectionName).GetKey(KeyName).
					AddValue(new CIniValueItem(Value, IsQuoted));
			}
			catch { throw; }
		}
		#endregion

		#region Read(string)
		/// <summary>Incorporates INI file. If INI file doesn't exist
		/// it will be created.</summary>
		/// <param name="FileName">Name and path of INI file.</param>
		protected void Read(string FileName) 
		{
			try 
			{
				ArrayList LeadingComments = null;
				bool IsCommentLine = false;
				bool IsSection = false;
				string SectionName = null;
				string KeyName = null;
				PARSE_POS ParsePos = PARSE_POS.A;

				StreamReader IniReader = null;
				try 
				{
					IniReader = new StreamReader((Stream)File.Open(this.FileName, 
						FileMode.OpenOrCreate, FileAccess.Read, FileShare.Read),
						this._TextEncoding);
					while (IniReader.Peek() > -1) 
					{
						CIniMatch IniMatch = new CIniMatch(IniReader.ReadLine());

						IsSection = IniMatch.MatchSectionName();
						if (IsSection) { ParsePos = PARSE_POS.A; }

						if (ParsePos == PARSE_POS.A)
						{
							// Comment lines before a section or section name line
							// respectively.
							if (IsSection)
							{
								AddSection(new CIniSectionItem(IniMatch.Value, 
									LeadingComments));
								SectionName = IniMatch.Value;
								// Add inline comment of section.
								if (IniMatch.MatchSectionComment())
								{
									GetSection(SectionName).Comment = IniMatch.Value;
								}

								LeadingComments = null;
								IsCommentLine = false;
								ParsePos = PARSE_POS.B;
							}
							else
							{
								IsCommentLine = true;
							}
						} 
						else if (ParsePos == PARSE_POS.B)
						{
							// After section name line.
							if (IniMatch.MatchKeyName())
							{
								// Key/value line.
								GetSection(SectionName).AddKey(
									new CIniKeyItem(IniMatch.Value, LeadingComments));
								KeyName = IniMatch.Value;
								// Add values of key.
								if (IniMatch.MatchKeyValues())
								{
									foreach (CIniValueItem ValueItem in IniMatch.Values)
									{
										GetSection(SectionName).GetKey(KeyName).AddValue(
											new CIniValueItem(ValueItem.Value, ValueItem.IsQuoted));
									}
								}
								// Add inline comment of key.
								if (IniMatch.MatchKeyComment())
								{
									GetSection(SectionName).GetKey(KeyName).Comment = 
										IniMatch.Value;
								}

								LeadingComments = null;
								IsCommentLine = false;
							}
							else
							{
								IsCommentLine = true;
							}
						} // else if

						if (IsCommentLine)
						{
							if (LeadingComments == null) 
							{
								LeadingComments = new ArrayList();
							}

							if (IniMatch.MatchComment())
							{
								// Comment line.
								LeadingComments.Add(IniMatch.Value);
							}
							else
							{
								// Empty line.
								LeadingComments.Add(null);
							}
						} // if

					} //while
				}
				finally 
				{
					if (IniReader != null) { IniReader.Close(); }
				}
			}
			catch { throw; }
		}
		#endregion

		#region Write()
		/// <summary>Write INI file.</summary>
		public void Write()
		{
			try
			{
				StreamWriter IniWriter = null;
				try
				{
					FileMode IniFileMode = FileMode.Truncate;
					if (!File.Exists(this.FileName)) 
					{ 
						IniFileMode = FileMode.Create;
					}
					IniWriter = new StreamWriter((Stream)File.Open(this.FileName, 
						IniFileMode, FileAccess.Write, FileShare.None), 
						this._TextEncoding);
					IniWriter.Write(this.ToString());
				}
				finally 
				{
					if (IniWriter != null) { IniWriter.Close(); }
				}
			}
			catch { throw; }
		}
		#endregion
	} // class

} // namespace