#region Copyright (c) Aquineo
/*
 * License
 * 
 * Aquineo disclaims all warranties with regard to this software, including all 
 * implied warranties of merchantability and fitness, in no event shall Aquineo 
 * be liable for any special, indirect or consequential damages or any damages 
 * whatsoever resulting from loss of use, data or profits, whether in an action 
 * of contract, negligence or other tortious action, arising out of or in 
 * connection with the use or performance of this software. 
 * 
 * Permission to use, copy, modify, and distribute this software for any purpose 
 * and without fee is hereby granted, subject to the following restrictions: 
 * 
 * 1. The origin of this software must not be misrepresented; you must not claim 
 * that you wrote the original software. If you use this software in a product, 
 * an acknowledgment (see the following) in the product documentation and this 
 * permission notice is required:
 *
 * Copyright  2004 Aquineo (www.aquineo.com)
 * 
 * 2. Altered source versions must be plainly marked as such, and must not be 
 * misrepresented as being the original software.
 * 
 * 3. This notice may not be removed or altered from any source distribution.
 * 
 * */
#endregion

/*
 * Classes: CIniKeyItem
 *
 * History: 
 *  2004-02-03, created
 * 
 * Remarks:
 *
 * ToDo:
 * */

using System;
using System.Collections;
using System.IO;
using System.Text;

namespace Planet.Utility.Ini
{

	/// <summary>This class manages a key as an element of a section.</summary>
	/// <remarks>Comment lines before a key name line are supported. Empty 
	/// lines are seen as comment lines assigned the value null.</remarks>
	public class CIniKeyItem
	{
		#region constructor
		/// <summary>Constructor</summary>
		/// <param name="Name">Name of the key.</param>
		/// <param name="LeadingComments">Comment lines, must be null if 
		/// comment is missing.</param>
		/// <exception cref="ArrayTypeMismatchException">One or more objects 
		/// in LeadingComments are not of type String.</exception>
		public CIniKeyItem(string Name, ArrayList LeadingComments)
		{
			try
			{
				this.Name = Name;

				if (LeadingComments == null) 
				{ 
					this.LeadingComments = new ArrayList(); 
				}
				else
				{
					foreach (object LeadingComment in LeadingComments)
					{
						if (LeadingComment != null && 
							  LeadingComment.GetType() != typeof(String))
						{
							throw new ArrayTypeMismatchException();
						}
					} // foreach
					this.LeadingComments = (ArrayList)LeadingComments.Clone();
				}
			}
			catch { throw; }
		}
		#endregion

		#region constants
		/// <summary>Character (not part of the key name) separating the key's 
		/// name from its values.</summary>
		public const char KEY_VALUE_DELIMITER = '=';
		/// <summary>Character separating the values from each other.</summary>
		public const char VALUE_DELIMITER = ',';
		/// <summary>Space character.</summary>
		protected const char SPACE = ' ';
		/// <summary>Characters not allowed inside of a key name. For 
		/// definition of whitespace not allowed see <see 
		/// cref="CIniValueItem.WHITESPACE_STRING"/>.</summary>
		public static readonly char[] INVALID_KEY_NAME_CHARS =
			new char[]{'=', ',', ';', '"', ' ', '\t', '\n', '\r', '\x000b', '\x000c'};
		#endregion

		#region members and properties
		/// <summary>See <see cref="Name"/>.</summary>
		protected string _Name = null;
		/// <value>Name of the key.</value>
		/// <exception cref="ArgumentException">Name is null or contains only
		/// whitespace, or contains characters defined in <see 
		/// cref="INVALID_KEY_NAME_CHARS"/>.</exception>
		public string Name
		{
			get
			{
				return _Name;
			}
			set
			{				
				if (!IsValidKeyName(value))
				{
					throw new ArgumentException();
				}
				_Name = value;
			} // set
		}

		/// <summary>Optional comment lines before a key name line.
		/// The array contains string values.</summary>
		protected ArrayList LeadingComments = null;

		/// <summary>Number of comment lines (see <see 
		/// cref="LeadingComments"/>).</summary>
		public int CountCommentLines
		{
			get
			{
				if (LeadingComments != null)
				{
					return LeadingComments.Count;
				}
				else
				{
					return -1;
				}
			} // get
		}

		/// <summary>See <see cref="Comment"/>.</summary>
		protected string _Comment = null;
		/// <summary>Optional inline comment after a key/value string.</summary>
		/// <exception cref="ArgumentException">Comment contains characters defined 
		/// in <see cref="CIni.INVALID_COMMENT_CHARS"/>.</exception>
		public string Comment 
		{
			get
			{
				return _Comment;
			}
			set
			{
				if (value != null && 
					  value.IndexOfAny(CIni.INVALID_COMMENT_CHARS) > -1)
				{
					throw new ArgumentException();
				}
				_Comment = value;
			}
		}

		/// <summary>Array of values, see <see cref="CIniValueItem"/>.
		/// </summary>
		protected ArrayList Values = new ArrayList();

		/// <summary>Number of values (see <see cref="Values"/>).</summary>
		public int CountValues
		{
			get
			{
				return Values.Count;
			}
		}
		#endregion

		#region IsValidKeyName(string)
		/// <summary>Checks whether key name is valid.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>True if key name is valid. False, name is null or 
		/// contains only whitespace, or contains characters defined in <see 
		/// cref="INVALID_KEY_NAME_CHARS"/>.</returns>
		public static bool IsValidKeyName(string KeyName)
		{
			try
			{
				// Don't trim key name, this is a checker!
				if (KeyName == null || KeyName == String.Empty ||
					  KeyName.IndexOfAny(INVALID_KEY_NAME_CHARS) > -1)
				{
					return false;
				}
				else
				{
					return true;
				}
			}
			catch { throw; }
		}
		#endregion

		#region GetCommentLine(int)
		/// <summary>Gets a comment line.</summary>
		/// <param name="Index">The zero-based index of a comment line. Maximum
		/// index is <see cref="CountCommentLines"/> - 1.</param>
		/// <returns>Comment line or null (empty line).</returns>
		/// <exception cref="NullReferenceException"><see 
		/// cref="LeadingComments"/> is null.</exception>
		/// <exception cref="ArgumentOutOfRangeException">Index is
		/// out of range.</exception>
		public string GetCommentLine(int Index)
		{
			try
			{
				if (LeadingComments == null)
				{
					throw new NullReferenceException();
				}
				if ((Index < 0) || (Index > CountCommentLines - 1))
				{
					throw new ArgumentOutOfRangeException();
				}

				if (LeadingComments[Index] == null)
				{
					return null;
				}
				else
				{
					return Convert.ToString(LeadingComments[Index]);
				}
			}
			catch { throw; }
		}
		#endregion

		#region AddCommentLine(string)
		/// <summary>Adds a comment line. Null is converted into an empty line.
		/// </summary>
		/// <param name="Comment">Comment line.</param>
		/// <exception cref="ArgumentException">Comment contains characters defined 
		/// in <see cref="CIni.INVALID_COMMENT_CHARS"/>.</exception>
		public void AddCommentLine(string Comment)
		{
			try
			{
				if (Comment != null && 
					  Comment.IndexOfAny(CIni.INVALID_COMMENT_CHARS) > -1)
				{
					throw new ArgumentException();
				}
				LeadingComments.Add(Comment);
			}
			catch { throw; }
		}
		#endregion

		#region ClearCommentLines()
		/// <summary>Removes all comment lines from
		/// <see cref="LeadingComments"/>.</summary>
		public void ClearCommentLines()
		{
			try
			{
				if (LeadingComments != null)
				{
					LeadingComments.Clear();
				}
			}
			catch { throw; }
		}
		#endregion

		#region GetValue(int, bool)
		/// <summary>Gets the string value of <see cref="Values"/>.</summary>
		/// <param name="IndexOfValue">The zero-based index of a value. Maximum
		/// index is <see cref="CountValues"/> - 1.</param>
		/// <param name="IgnoreQuoteFlag">True will return the sole string value
		/// not enclosed in <see cref="CIniValueItem.QUOTE_DELIMITER"/>. 
		/// False encloses the value in <see cref="CIniValueItem.QUOTE_DELIMITER"/>
		/// if the quote flag is set.</param>
		/// <returns>String value.</returns>
		/// <exception cref="ArgumentOutOfRangeException">IndexOfValue is
		/// out of range.</exception>
		public string GetValue(int IndexOfValue, bool IgnoreQuoteFlag)
		{
			try
			{
				if ((IndexOfValue < 0) || (IndexOfValue > CountValues - 1))
				{
					throw new ArgumentOutOfRangeException();
				}

				if (IgnoreQuoteFlag || !((CIniValueItem)Values[IndexOfValue]).IsQuoted)
				{
					return ((CIniValueItem)Values[IndexOfValue]).Value;
				}
				else
				{
					return CIniValueItem.QUOTE_DELIMITER +
						((CIniValueItem)Values[IndexOfValue]).Value +
						CIniValueItem.QUOTE_DELIMITER;
				} // else
			}
			catch { throw; }
		}
		#endregion

		#region AddValue(CIniValueItem)
		/// <summary>Adds a value (see <see cref="CIniValueItem"/>).</summary>
		/// <param name="ValueItem">Value to be added (see <see 
		/// cref="CIniValueItem"/>).</param>
		/// <exception cref="ArgumentNullException">Null as an argument.</exception>
		public void AddValue(CIniValueItem ValueItem)
		{
			try
			{
				if (ValueItem == null) 
				{
					throw new ArgumentNullException();
				}
				Values.Add(ValueItem);
			}
			catch { throw; }
		}
		#endregion

		#region ClearValues()
		/// <summary>Deletes all values.</summary>
		public void ClearValues()
		{
			try
			{
				Values.Clear();
			}
			catch { throw; }
		}
		#endregion

		#region ToString()
		/// <summary>Returns entire key including comment lines and
		/// key/value pair. The section may contain or may not contain comments or
		/// keys.</summary>
		/// <returns>Key element as a string.</returns>
		public override string ToString()
		{
			try
			{
				StringBuilder KeyItem = new StringBuilder(String.Empty);

				// Comment line.
				foreach (string LeadingComment in this.LeadingComments)
				{
					if (LeadingComment == null)
					{
						KeyItem.Append(String.Empty);
					}
					else
					{
						KeyItem.Append(CIni.COMMENT_DELIMITER);
						KeyItem.Append(LeadingComment);
					}
					KeyItem.Append(CIni.GetLineTerminator());
				} // foreach

				KeyItem.Append(this.Name);
				KeyItem.Append(KEY_VALUE_DELIMITER);

				if (Values != null && Values.Count > 0)
				{
					for (int i = 0; i < Values.Count - 1; ++i)
					{
						if (((CIniValueItem)Values[i]).IsQuoted)
						{
							KeyItem.Append(
								CIniValueItem.QUOTE_DELIMITER +
								((CIniValueItem)Values[i]).Value + 
								CIniValueItem.QUOTE_DELIMITER);
						}
						else
						{
							KeyItem.Append(((CIniValueItem)Values[i]).Value);
						}
						KeyItem.Append(VALUE_DELIMITER);
					} // for
					// Add last value.
					if (((CIniValueItem)Values[Values.Count - 1]).IsQuoted)
					{
						KeyItem.Append(
							CIniValueItem.QUOTE_DELIMITER +
							((CIniValueItem)Values[Values.Count - 1]).Value +
							CIniValueItem.QUOTE_DELIMITER);
					}
					else
					{
						KeyItem.Append(((CIniValueItem)Values[Values.Count - 1]).Value);
					}
				} // if

				if (this.Comment != null)
				{
					KeyItem.Append(SPACE);
					KeyItem.Append(CIni.COMMENT_DELIMITER);
					KeyItem.Append(this.Comment);
				}

				return KeyItem.ToString();
			}
			catch { throw; }
		}
		#endregion
	} // class

} // namespace
