#region Copyright (c) Aquineo
/*
 * License
 * 
 * Aquineo disclaims all warranties with regard to this software, including all 
 * implied warranties of merchantability and fitness, in no event shall Aquineo 
 * be liable for any special, indirect or consequential damages or any damages 
 * whatsoever resulting from loss of use, data or profits, whether in an action 
 * of contract, negligence or other tortious action, arising out of or in 
 * connection with the use or performance of this software. 
 * 
 * Permission to use, copy, modify, and distribute this software for any purpose 
 * and without fee is hereby granted, subject to the following restrictions: 
 * 
 * 1. The origin of this software must not be misrepresented; you must not claim 
 * that you wrote the original software. If you use this software in a product, 
 * an acknowledgment (see the following) in the product documentation and this 
 * permission notice is required:
 *
 * Copyright  2004 Aquineo (www.aquineo.com)
 * 
 * 2. Altered source versions must be plainly marked as such, and must not be 
 * misrepresented as being the original software.
 * 
 * 3. This notice may not be removed or altered from any source distribution.
 * 
 * */
#endregion

/*
 * Classes: CIniMatch
 *
 * History: 
 *  2004-02-14, created
 * 
 * Remarks:
 *
 * ToDo:
 * */

using System;
using System.Collections;
using System.Text;

namespace Planet.Utility.Ini
{

	/// <summary>Searches an INI file line and keeps the result.</summary>
	/// <remarks>The searches of this class could have been implemented
	/// using regular expressions but tests made clear that performance and 
	/// efficiency is quite bad.
	/// </remarks>
	public class CIniMatch
	{
		#region constructor
		/// <summary>Constructor</summary>
		/// <param name="Line">The string to search for a match.</param>
		public CIniMatch(string Line)
		{
			try
			{
				// Assignation of parameter Line initializes the class.
				this.Line = Line;
			}
			catch { throw; }
		}
		#endregion

		#region members and properties
		/// <summary>See <see cref="Line"/>.</summary>
		protected string _Line;
		/// <summary>Input string to search in (musn't be null).</summary>
		/// <exception cref="NullReferenceException">String is null.</exception>
		public string Line
		{
			get
			{
				return _Line;
			}
			set
			{
				try
				{
					// Wipe out last search result.
					Init();
					if (value != null)
					{
						// Search is tolerant towards whitespace at the beginning 
						// of the string.
						_Line = value.TrimStart(null);
					}
					else { throw new NullReferenceException(); }
				}
				catch { throw; }
			}
		}

		/// <summary>See <see cref="Success"/>.</summary>
		protected bool _Success;
		/// <summary>Indicates whether the match is successful.</summary>
		public bool Success
		{
			get
			{
				return _Success;
			}
		}

		/// <summary>See <see cref="Index"/>.</summary>
		protected int _Index;
		/// <summary>The zero-based index in the original string <see cref="Line"/>
		/// where the first character of the captured substring <see cref="Value"/> 
		/// was found; otherwise, -1. Index of empty string is always -1.</summary>
		public int Index
		{
			get
			{
				return _Index;
			}
		}

		/// <summary>Gets the length of the captured substring; otherwise, 
		/// -1.</summary>
		public int Length
		{
			get
			{
				if (this._Value == null)
				{
					return -1;
				}
				else
				{
					return this._Value.Length;
				}
			}
		}

		/// <summary>See <see cref="Value"/>.</summary>
		protected string _Value;
		/// <summary>Gets the captured substring from the input string
		/// <see cref="Line"/>. For objects matched by <see cref="MatchKeyValues()"/> 
		/// see <see cref="Values"/>.</summary>
		public string Value
		{
			get
			{
				return _Value;
			}
		}

		/// <summary>See <see cref="Values"/>.</summary>
		protected ArrayList _Values;
		/// <summary>A collection of <see cref="CIniValueItem"/> objects matched 
		/// by <see cref="MatchKeyValues()"/>. All other search methods capture
		/// results in <see cref="Value"/>.</summary>
		public ArrayList Values
		{
			get
			{
				return _Values;
			}
		}
		#endregion

		#region Init()
		/// <summary>Initialize class members (wipe out results of
		/// last search).</summary>
		protected void Init()
		{
			try
			{
				this._Value = null;
				this._Values = null;
				this._Index = -1;
				this._Success = false;
			}
			catch { throw; }
		}
		#endregion

		#region MatchSectionName()
		/// <summary>Searches the specified input string for the occurrence of
		/// a section name.</summary>
		/// <returns>Indicates whether the search was successful. 
		/// The substring will be captured in <see cref="Value"/>,
		/// if successful.</returns>
		public bool MatchSectionName()
		{
			try
			{
				Init();
				
				int IndexOfNamePrefix = 
					this._Line.IndexOf(CIniSectionItem.SECTION_NAME_PREFIX);
				int IndexOfNamePostfix = 
					this._Line.IndexOf(CIniSectionItem.SECTION_NAME_POSTFIX);

				// Must contain at least one character.
				if (IndexOfNamePrefix == 0 && IndexOfNamePostfix > 1)
				{
					// Extract the name and eliminate surrounding whitespace.
					this._Value = this._Line.Substring(1, IndexOfNamePostfix - 1).Trim();
					if (this._Value.Length > 0 && 
						  this._Value.IndexOfAny(CIniSectionItem.INVALID_SECTION_NAME_CHARS) == -1)
					{
						this._Index = this._Line.IndexOf(this._Value);
						this._Success = true;
					}
					if (!this._Success) { this._Value = null; }
				} // if				

				return this._Success;
			}
			catch { throw; }
		}
		#endregion

		#region MatchKeyName()
		/// <summary>Searches the specified input string for the occurrence of
		/// a key name.</summary>
		/// <returns>Indicates whether the search was successful.
		/// The substring will be captured in <see cref="Value"/>,
		/// if successful.</returns>
		public bool MatchKeyName()
		{
			try
			{
				Init();
				
				int IndexOfKeyValueDelimiter = 
					this._Line.IndexOf(CIniKeyItem.KEY_VALUE_DELIMITER);

				// Must contain at least one character.
				if (IndexOfKeyValueDelimiter > 0)
				{
					// Extract the name and eliminate surrounding whitespace. 
					// Be aware that beginning whitespace has been deleted in advance.
					this._Value = this._Line.Substring(0, IndexOfKeyValueDelimiter).Trim();
					if (this._Value.Length > 0 && 
						  this._Value.IndexOfAny(CIniKeyItem.INVALID_KEY_NAME_CHARS) == -1)
					{
						this._Index = 0;
						this._Success = true;
					}
					if (!this._Success) { this._Value = null; }
				} // if

				return this._Success;
			}
			catch { throw; }
		}
		#endregion

		#region MatchComment()
		/// <summary>Searches the specified input string for the occurrence of
		/// a comment line.</summary>
		/// <returns>Indicates whether the search was successful.
		/// The substring will be captured in <see cref="Value"/>,
		/// if successful.</returns>
		public bool MatchComment()
		{
			try
			{
				Init();
				
				int IndexOfCommentDelimiter = 
					this._Line.IndexOf(CIni.COMMENT_DELIMITER);
				
				if (IndexOfCommentDelimiter == 0)
				{
					// Whitespace is part of a comment, don't skip.
					if (this._Line.Length == 1)
					{ 
						this._Value = String.Empty;
					}
					else
					{
						this._Value = this._Line.Substring(1);
						this._Index = 1;
					}					
					this._Success = true;
				} // if

				return this._Success;
			}
			catch { throw; }
		}
		#endregion

		#region MatchSectionComment()
		/// <summary>Searches the specified input string, containing a section 
		/// name, for the occurrence of a comment.</summary>
		/// <returns>Indicates whether the search was successful.
		/// The substring will be captured in <see cref="Value"/>,
		/// if successful.</returns>
		public bool MatchSectionComment()
		{
			try
			{				
				if (MatchSectionName())
				{
					// Remember start index and length of the section name before Init()
					// will reinitialize it.
					int IndexOfSectionName = this._Index;
					int SectionNameLength = this.Length;

					Init();

					// Start search after section name.
					int IndexOfCommentDelimiter = this._Line.IndexOf(
						CIni.COMMENT_DELIMITER, IndexOfSectionName + SectionNameLength);

					if (IndexOfCommentDelimiter > -1)
					{
						// Whitespace is part of a comment: don't skip.
						if (this._Line.Length == IndexOfCommentDelimiter + 1)
						{ 
							this._Value = String.Empty;							
						}
						else
						{
							this._Value = this._Line.Substring(IndexOfCommentDelimiter + 1);
							this._Index = IndexOfCommentDelimiter + 1;
						} 
						this._Success = true;
					} // if
				} // if

				return this._Success;
			}
			catch { throw; }
		}
		#endregion

		#region MatchKeyValues()
		/// <summary>Searches the specified input string for the occurrence of
		/// values of a key/value pair.</summary>
		/// <returns>Indicates whether the search was successful. <see cref="Values"/>
		/// contains the list of separated values as <see cref="CIniValueItem"/>
		/// objects, if successful. The unmodified value string is 
		/// stored in <see cref="Value"/>.  
		/// <see cref="Length"/> is meaningless for this search, it indicates
		/// the length of the unmodified value string.</returns>
		public bool MatchKeyValues()
		{
			try
			{				
				if (MatchKeyName())
				{					
					// Remember start index and length of the key name before Init()
					// will reinitialize it.
					int IndexOfKeyName = this._Index;
					int KeyNameLength = this.Length;

					Init();

					// Initial value of IndexOfValuesBegin must match with
					// this._Index in Init().
					int IndexOfValuesBegin = -1;
					int IndexOfValuesEnd = -1;

					bool IsQuoted = false;
					bool IsValue = false;
					StringBuilder CurrentValue = null;
					this._Values = new ArrayList();

					int i = this._Line.IndexOf(CIniKeyItem.KEY_VALUE_DELIMITER, 
						IndexOfKeyName + KeyNameLength) + 1;
					if (i >= this._Line.Length)
					{
						// End of line after key name means String.Empty (this would
						// not have been handled in the for loop).
						this._Values.Add(new CIniValueItem(String.Empty, false));
					}

					for (; i < this._Line.Length; ++i)
					{						
						if (!IsValue)
						{							
							if (!IsQuoted) // Last value wasn't quoted.
							{
								if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER ||
									  this._Line[i] == CIni.COMMENT_DELIMITER ||
									  (i == this._Line.Length - 1 && 
									  CIniValueItem.WHITESPACE_STRING.IndexOf(this._Line[i]) > -1))
								{
									// This._Line[i - 1] was CIniKeyItem.VALUE_DELIMITER,
									// handle the following situations:
									// a) Comma/equal sign followed by an additional comma 
									//    or semicolon;
									// b) a line ends with a whitespace character (in this 
									//    situation).
									this._Values.Add(new CIniValueItem(String.Empty, false));
								}
							}

							// a) New value
							// b) Skip whitespace and comma at the ending. Comma is redundant in
							//    this situation because double quotation marks closed the 
							//    value already.
							if (CIniValueItem.WHITESPACE_STRING.IndexOf(this._Line[i]) == -1 &&
									this._Line[i] != CIniKeyItem.VALUE_DELIMITER &&
									this._Line[i] != CIni.COMMENT_DELIMITER)
							{
								// New value.
								CurrentValue = new StringBuilder(String.Empty);	
								IsValue = true;
								if (this._Line[i] == CIniValueItem.QUOTE_DELIMITER)
								{
									// Quoted value, quotes aren't part of the value.
									IsQuoted = true;
									if (i == this._Line.Length - 1)
									{
										// Empty string as closing quote is missing:
										// key="
										this._Values.Add(new CIniValueItem(String.Empty, true));
										// No characters will follow, remember end of values index.
										IndexOfValuesEnd = i;
									}
								}
								else
								{
									// Value without quotes.
									IsQuoted = false;
									CurrentValue.Append(this._Line[i]);
									if (i == this._Line.Length - 1)
									{
										// One character only...
										this._Values.Add(new CIniValueItem(CurrentValue.ToString(), false));
										// No characters will follow, remember end of values index.
										IndexOfValuesEnd = i;
									} // if
								} // else
							}
							else if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER)
							{
								IsQuoted = false;
								// If only whitespace follows remember last character position.
								IndexOfValuesEnd = i;
								if (i == this._Line.Length - 1)
								{
									// Comma is last character of value string, add additional
									// empty string.
									this._Values.Add(new CIniValueItem(String.Empty, false));
								}
							}
							else if (this._Line[i] == CIni.COMMENT_DELIMITER)
							{
								break;
							}

							// Make a note of beginning of values list.
							if (IndexOfValuesBegin == -1 && 
								  CIniValueItem.WHITESPACE_STRING.IndexOf(this._Line[i]) == -1) 
							{ 
								IndexOfValuesBegin = i; 
							}
						} // if (!IsValue)
						else
						{
							if (IsQuoted)
							{	
								// Quoted value.

								if (this._Line[i] == CIniValueItem.QUOTE_DELIMITER)
								{
									// Not encoded quote or end of line closes quoted value.
									if (this._Line[i - 1] != CIniValueItem.ESCAPE ||
										  i == this._Line.Length - 1)
									{
										// Parser is tolerant towards the following situation:
										// key="...any_value\" - last character of quoted value is a quote!
										if (i == this._Line.Length - 1 &&
											  this._Line[i - 1] == CIniValueItem.ESCAPE)
										{
											CurrentValue.Append(CIniValueItem.QUOTE_DELIMITER);
										}
										this._Values.Add(new CIniValueItem(CurrentValue.ToString(), true));
										IndexOfValuesEnd = i;
										IsValue = false;
									}
									else
									{
										CurrentValue.Append(this._Line[i]);
									}
								} 
								else
								{
									CurrentValue.Append(this._Line[i]);
									// Although closing quote is missing end of line enforces 
									// saving of the value.
									if (i == this._Line.Length - 1)
									{
										this._Values.Add(new CIniValueItem(CurrentValue.ToString(), true));
										IndexOfValuesEnd = i;
										IsValue = false;
									}
								} // else
							} // if (IsQuoted)
							else
							{
								// Not quoted value.

								// Comma, semicolon, or end of line closes value.
								if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER ||
									  this._Line[i] == CIni.COMMENT_DELIMITER ||
										i == this._Line.Length - 1)
								{									
									// Add last character in line if not whitespace.
									if (this._Line[i] != CIniKeyItem.VALUE_DELIMITER &&
										  this._Line[i] != CIni.COMMENT_DELIMITER &&
										  i == this._Line.Length - 1)
									{
										CurrentValue.Append(this._Line[i]);
									}

									// Remove whitespace at the end of the value.
									if (CurrentValue.Length > 0)
									{										
										int IndexOfLastChar = - 1;
										int j;
										if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER ||
											  this._Line[i] == CIni.COMMENT_DELIMITER)
										{
											j = i - 1;
										}
										else
										{
											j = i;
										}
										int ReverseIndex = 0;
										for (; ReverseIndex < CurrentValue.Length; ++ReverseIndex)
										{
											if (CIniValueItem.WHITESPACE_STRING.IndexOf(
												  this._Line[j - ReverseIndex]) == -1)
											{
												// Last not-whitespace character found.
												IndexOfLastChar = CurrentValue.Length - 1 - ReverseIndex;
												break;
											} // if
										} // for

										// If whitespace exists at the end, remove it.
										CurrentValue.Remove(IndexOfLastChar + 1, ReverseIndex);
										if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER)
										{
											IndexOfValuesEnd = i;
										}
										else
										{
											IndexOfValuesEnd = j - ReverseIndex;
										}
									} // if

									this._Values.Add(new CIniValueItem(CurrentValue.ToString(), false));
									IsValue = false;

									if (this._Line[i] == CIniKeyItem.VALUE_DELIMITER &&
										  i == this._Line.Length - 1)
									{
										// Comma is last character of value string, add additional
										// empty string.
										this._Values.Add(new CIniValueItem(String.Empty, false));
									}

									if (this._Line[i] == CIni.COMMENT_DELIMITER) { break; }
								}
								else
								{
									// Add character inside of a not quoted value.
									CurrentValue.Append(this._Line[i]);
								}
							} // else
						} // else
					} // foreach

					if (this._Values.Count > 0) 
					{ 
						this._Success = true; 
						this._Index = IndexOfValuesBegin;
						if (IndexOfValuesBegin != -1 &&
							  IndexOfValuesEnd >= IndexOfValuesBegin) 
						{
							this._Value = this._Line.Substring(IndexOfValuesBegin,
								IndexOfValuesEnd - IndexOfValuesBegin + 1);
						}
						else
						{
							this._Value = String.Empty;
						}
					}
				} // if

				return this._Success;
			}
			catch { throw; }
		}
		#endregion

		#region MatchKeyComment()
		/// <summary>Searches the specified input string, containing a key 
		/// name, for the occurrence of a comment.</summary>
		/// <returns>Indicates whether the search was successful.
		/// The substring will be captured in <see cref="Value"/>,
		/// if successful.</returns>
		public bool MatchKeyComment()
		{
			try
			{
				if (MatchKeyName())
				{
					// Remember start index and length before Init() 
					// will reinitialize it.
					int IndexOfKeyValues = this._Index;
					int KeyValuesLength = this.Length;

					if (MatchKeyValues() && this.Length > 0)
					{
						IndexOfKeyValues = this._Index;
						KeyValuesLength = this.Length;
					}

					Init();

					// Start search after values.
					int IndexOfCommentDelimiter = this._Line.IndexOf(
						CIni.COMMENT_DELIMITER, IndexOfKeyValues + KeyValuesLength);

					if (IndexOfCommentDelimiter > -1)
					{
						// Whitespace is part of a comment: don't skip.
						if (this._Line.Length == IndexOfCommentDelimiter + 1)
						{ 
							this._Value = String.Empty;
						}
						else
						{
							this._Value = this._Line.Substring(IndexOfCommentDelimiter + 1);
							this._Index = IndexOfCommentDelimiter + 1;
						}						
						this._Success = true;
					} // if
				} // if
				
				return this._Success;
			}
			catch { throw; }
		}
		#endregion
	} // class

} // namespace
