#region Copyright (c) Aquineo
/*
 * License
 * 
 * Aquineo disclaims all warranties with regard to this software, including all 
 * implied warranties of merchantability and fitness, in no event shall Aquineo 
 * be liable for any special, indirect or consequential damages or any damages 
 * whatsoever resulting from loss of use, data or profits, whether in an action 
 * of contract, negligence or other tortious action, arising out of or in 
 * connection with the use or performance of this software. 
 * 
 * Permission to use, copy, modify, and distribute this software for any purpose 
 * and without fee is hereby granted, subject to the following restrictions: 
 * 
 * 1. The origin of this software must not be misrepresented; you must not claim 
 * that you wrote the original software. If you use this software in a product, 
 * an acknowledgment (see the following) in the product documentation and this 
 * permission notice is required:
 *
 * Copyright  2004 Aquineo (www.aquineo.com)
 * 
 * 2. Altered source versions must be plainly marked as such, and must not be 
 * misrepresented as being the original software.
 * 
 * 3. This notice may not be removed or altered from any source distribution.
 * 
 * */
#endregion

/*
 * Classes: CIniSectionItem
 *
 * History: 
 *  2004-02-03, created
 * 
 * Remarks:
 *
 * ToDo:
 * */

using System;
using System.Collections;
using System.IO;
using System.Text;

namespace Planet.Utility.Ini
{

	/// <summary>This class manages a section as an element of an INI file.
	/// </summary>
	/// <remarks>Comment lines before the section name line are supported. 
	/// Empty lines are seen as comment lines assigned the value null.</remarks>
	public class CIniSectionItem
	{
		#region constructor
		/// <summary>Constructor</summary>
		/// <param name="Name">Name of the section.</param>
		/// <param name="LeadingComments">Comment lines, must be null if 
		/// comment is missing.</param>
		/// <exception cref="ArrayTypeMismatchException">One or more objects 
		/// in LeadingComments are not of type String.</exception>
		public CIniSectionItem(string Name, ArrayList LeadingComments)
		{
			try
			{
				this.Name = Name;		
	
				if (LeadingComments == null) 
				{ 
					this.LeadingComments = new ArrayList(); 
				}
				else
				{
					foreach (object LeadingComment in LeadingComments)
					{
						if (LeadingComment != null && 
							  LeadingComment.GetType() != typeof(String))
						{
							throw new ArrayTypeMismatchException();
						}
					} // foreach
					this.LeadingComments = (ArrayList)LeadingComments.Clone();
				}
			}
			catch { throw; }
		}
		#endregion

		#region constants
		/// <summary>Character (not part of the section name) delimiting
		/// the section's name on the left.</summary>
		public const char SECTION_NAME_PREFIX = '[';
		/// <summary>Character (not part of the section name) delimiting
		/// the section's name on the right.</summary>
		public const char SECTION_NAME_POSTFIX = ']';
		/// <summary>Space character.</summary>
		protected const char SPACE = ' ';
		/// <summary>Characters not allowed inside of a section name. For 
		/// definition of whitespace not allowed see <see 
		/// cref="CIniValueItem.WHITESPACE_STRING"/>.</summary>
		public static readonly char[] INVALID_SECTION_NAME_CHARS = 
			new char[]{'[', ']', ',', ';', '"', ' ', '\t', '\n', '\r', '\x000b', '\x000c'};
		#endregion

		#region members and properties
		/// <summary>See <see cref="Name"/>.</summary>
		protected string _Name = null;
		/// <value>Name of the section.</value>
		/// <exception cref="ArgumentException">Name is null or contains only
		/// whitespace, or contains characters defined in <see 
		/// cref="INVALID_SECTION_NAME_CHARS"/>.</exception>
		public string Name
		{
			get
			{
				return _Name;
			}
			set
			{
				if (!IsValidSectionName(value))
				{
					throw new ArgumentException();
				}
				_Name = value;
			} // set
		}

		/// <summary>Optional comment lines before a section name line.
		/// The array contains string values.</summary>
		protected ArrayList LeadingComments = null;

		/// <summary>Number of comment lines (see <see 
		/// cref="LeadingComments"/>).</summary>
		public int CountCommentLines
		{
			get
			{
				if (LeadingComments != null)
				{
					return LeadingComments.Count;
				}
				else
				{
					return -1;
				}
			} // get
		}

		/// <summary>See <see cref="Comment"/>.</summary>
		protected string _Comment = null;
		/// <summary>Optional inline comment after the section name string.</summary>
		/// <exception cref="ArgumentException">Comment contains characters defined 
		/// in <see cref="CIni.INVALID_COMMENT_CHARS"/>.</exception>
		public string Comment 
		{
			get
			{
				return _Comment;
			}
			set
			{
				if (value != null && 
					  value.IndexOfAny(CIni.INVALID_COMMENT_CHARS) > -1)
				{
					throw new ArgumentException();
				}
				_Comment = value;
			}
		}

		/// <summary>List of key elements, see <see cref="CIniKeyItem"/>.</summary>
		protected ArrayList Keys = new ArrayList();
		#endregion

		#region IsValidSectionName(string)
		/// <summary>Checks whether section name is valid.</summary>
		/// <param name="SectionName">Name of section.</param>
		/// <returns>True if section name is valid. False, name is null or 
		/// contains only whitespace, or contains characters defined in <see 
		/// cref="INVALID_SECTION_NAME_CHARS"/>.</returns>
		public static bool IsValidSectionName(string SectionName)
		{
			try
			{
				// Don't trim section name, this is a checker!
				if (SectionName == null || SectionName == String.Empty ||
					  SectionName.IndexOfAny(INVALID_SECTION_NAME_CHARS) > -1)
				{
					return false;
				}
				else
				{
					return true;
				}
			}
			catch { throw; }
		}
		#endregion

		#region IndexOfKey(string)
		/// <summary>Searches for the first occurrence of a key.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>The zero-based index of the first occurrence of 
		/// a key within the entire ArrayList <see cref="Keys"/>, 
		/// if found; otherwise, -1.</returns>
		public int IndexOfKey(string KeyName)
		{
			try
			{
				int Index = -1;
				for (int i = 0; i < Keys.Count; ++i)
				{
					if (((CIniKeyItem)Keys[i]).Name == KeyName)
					{
						Index = i;
						break;
					}
				}
				return Index;
			}
			catch { throw; }
		}
		#endregion

		#region IsKey(string)
		/// <summary>Checks existence of a key.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>True if key exists.</returns>
		public bool IsKey(string KeyName)
		{
			try
			{
				if (!CIniKeyItem.IsValidKeyName(KeyName))
				{
					return false;
				}
				else
				{
					int Index = IndexOfKey(KeyName);
					if (Index > -1 && Keys[Index] != null)
					{
						return true;
					}
					else
					{
						return false;
					}
				} // else
			}
			catch { throw; }
		}
		#endregion

		#region GetCommentLine(int)
		/// <summary>Gets a comment line.</summary>
		/// <param name="Index">The zero-based index of a comment line. Maximum
		/// index is <see cref="CountCommentLines"/> - 1.</param>
		/// <returns>Comment line or null (empty line).</returns>
		/// <exception cref="NullReferenceException"><see 
		/// cref="LeadingComments"/> is null.</exception>
		/// <exception cref="ArgumentOutOfRangeException">Index is
		/// out of range.</exception>
		public string GetCommentLine(int Index)
		{
			try
			{
				if (LeadingComments == null)
				{
					throw new NullReferenceException();
				}
				if ((Index < 0) || (Index > CountCommentLines - 1))
				{
					throw new ArgumentOutOfRangeException();
				}

				if (LeadingComments[Index] == null)
				{
					return null;
				}
				else
				{
					return Convert.ToString(LeadingComments[Index]);
				}
			}
			catch { throw; }
		}
		#endregion

		#region AddCommentLine(string)
		/// <summary>Adds a comment line. Null is converted into an empty line.
		/// </summary>
		/// <param name="Comment">Comment line.</param>
		/// <exception cref="ArgumentException">Comment contains characters defined 
		/// in <see cref="CIni.INVALID_COMMENT_CHARS"/>.</exception>
		public void AddCommentLine(string Comment)
		{
			try
			{				
				if (Comment != null && 
					  Comment.IndexOfAny(CIni.INVALID_COMMENT_CHARS) > -1)
				{
					throw new ArgumentException();
				}
				LeadingComments.Add(Comment);
			}
			catch { throw; }
		}
		#endregion

		#region ClearCommentLines()
		/// <summary>Removes all comment lines from
		/// <see cref="LeadingComments"/>.</summary>
		public void ClearCommentLines()
		{
			try
			{
				if (LeadingComments != null)
				{
					LeadingComments.Clear();
				}
			}
			catch { throw; }
		}
		#endregion

		#region AddKey(string, ArrayList)
		/// <summary>Adds a key and its optional comment lines. Overrides
		/// existing keys with the same name.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <param name="LeadingComments">Comment lines, must be null if 
		/// comment is missing.</param>
		/// <returns>Returns a reference to the added key element.</returns>
		/// <exception cref="ArgumentException">Invalid key name,
		/// see <see cref="CIniKeyItem.IsValidKeyName(string)"/>.</exception>
		/// <exception cref="ArrayTypeMismatchException">One or more objects 
		/// in LeadingComments are not of type String.</exception>		
		public CIniKeyItem AddKey(string KeyName, 
			ArrayList LeadingComments)
		{
			try
			{
				if (!CIniKeyItem.IsValidKeyName(KeyName)) 
				{ 
					throw new ArgumentException(); 
				}

				// Override existing key.
				int Index = IndexOfKey(KeyName);
				if (Index > -1)
				{
					// Key exists, override key.
					Keys[Index] = new CIniKeyItem(KeyName, LeadingComments);
				}
				else
				{
					// Create new key.
					Index = Keys.Add(new CIniKeyItem(KeyName, LeadingComments));
				}
				
				return (CIniKeyItem)Keys[Index];
			}
			catch { throw; }
		}
		#endregion

		#region AddKey(CIniKeyItem)
		/// <summary>Adds a key element, overrides existing keys with the
		/// same name.</summary>
		/// <param name="KeyItem">Key element to be added.</param>
		/// <returns>Returns a reference to the added key element.</returns>
		/// <exception cref="ArgumentException">Invalid key name,
		/// see <see cref="CIniKeyItem.IsValidKeyName(string)"/>.</exception>
		/// <exception cref="ArgumentNullException">Null as an argument.</exception>
		public CIniKeyItem AddKey(CIniKeyItem KeyItem)
		{
			try
			{
				if (KeyItem == null)
				{
					throw new ArgumentNullException();
				}
				if (!CIniKeyItem.IsValidKeyName(KeyItem.Name)) 
				{ 
					throw new ArgumentException(); 
				}

				// Override existing key.
				int Index = IndexOfKey(KeyItem.Name);
				if (Index > -1)
				{
					// Key exists, override key.
					Keys[Index] = KeyItem;
				}
				else
				{
					// Create new key.
					Index = Keys.Add(KeyItem);
				}
				
				return (CIniKeyItem)Keys[Index];
			}
			catch { throw; }
		}
		#endregion

		#region GetKey(string)
		/// <summary>Search for the first occurrence of a key.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <returns>Key, if found; otherwise, null.</returns>
		public CIniKeyItem GetKey(string KeyName)
		{
			try
			{
				int Index = IndexOfKey(KeyName);
				if (Index > -1)
				{
					// Element exists, value is not null.
					return (CIniKeyItem)Keys[Index];
				}
				else
				{
					// Element doesn't exist.
					return null;
				}
			}
			catch { throw; }
		}
		#endregion

		#region RemoveKey(string)
		/// <summary>Deletes a key. If the name of the key is valid but doesn't 
		/// exist no operation is performed.</summary>
		/// <param name="KeyName">Name of a key.</param>
		/// <exception cref="ArgumentException">Invalid key name,
		/// see <see cref="CIniKeyItem.IsValidKeyName(string)"/>.</exception>
		public void RemoveKey(string KeyName)
		{
			try
			{
				if (!CIniKeyItem.IsValidKeyName(KeyName)) 
				{
					throw new ArgumentException(); 
				}

				if (Keys.Count > 0)
				{
					int Index = IndexOfKey(KeyName);
					if (Index > -1) { Keys.RemoveAt(Index); }
				}
			}
			catch { throw; }
		}
		#endregion

		#region ToString()
		/// <summary>Returns entire section including comment lines and
		/// keys. The section may contain or not contain comments or
		/// keys.</summary>
		/// <returns>Section element as a string.</returns>
		public override string ToString()
		{
			try
			{
				StringBuilder SectionItem = new StringBuilder(String.Empty);

				// Comment line.
				foreach (string LeadingComment in this.LeadingComments)
				{
					if (LeadingComment == null)
					{
						SectionItem.Append(String.Empty);
					}
					else
					{
						SectionItem.Append(CIni.COMMENT_DELIMITER);
						SectionItem.Append(LeadingComment);
					}
					SectionItem.Append(CIni.GetLineTerminator());
				} // foreach

				SectionItem.Append(SECTION_NAME_PREFIX);
				SectionItem.Append(this.Name);
				SectionItem.Append(SECTION_NAME_POSTFIX);

				if (this.Comment != null)
				{
					SectionItem.Append(SPACE);
					SectionItem.Append(CIni.COMMENT_DELIMITER);
					SectionItem.Append(this.Comment);
				}

				if (Keys != null)
				{
					foreach (CIniKeyItem Key in Keys)
					{
						SectionItem.Append(CIni.GetLineTerminator());
						SectionItem.Append(Key.ToString());
					} 
				} // if

				return SectionItem.ToString();
			}
			catch { throw; }
		}
		#endregion
	} // class

} // namespace
