#region Copyright (c) Aquineo
/*
 * License
 * 
 * Aquineo disclaims all warranties with regard to this software, including all 
 * implied warranties of merchantability and fitness, in no event shall Aquineo 
 * be liable for any special, indirect or consequential damages or any damages 
 * whatsoever resulting from loss of use, data or profits, whether in an action 
 * of contract, negligence or other tortious action, arising out of or in 
 * connection with the use or performance of this software. 
 * 
 * Permission to use, copy, modify, and distribute this software for any purpose 
 * and without fee is hereby granted, subject to the following restrictions: 
 * 
 * 1. The origin of this software must not be misrepresented; you must not claim 
 * that you wrote the original software. If you use this software in a product, 
 * an acknowledgment (see the following) in the product documentation and this 
 * permission notice is required:
 *
 * Copyright  2004 Aquineo (www.aquineo.com)
 * 
 * 2. Altered source versions must be plainly marked as such, and must not be 
 * misrepresented as being the original software.
 * 
 * 3. This notice may not be removed or altered from any source distribution.
 * 
 * */
#endregion

/*
 * Classes: CIniValueItem
 *
 * History: 
 *  2004-02-03, created
 * 
 * Remarks:
 *
 * ToDo:
 * */

using System;
using System.Collections;
using System.IO;
using System.Text;

namespace Planet.Utility.Ini
{

	/// <summary>This class manages a value as an element of a key.</summary>
	public class CIniValueItem
	{
		#region constructor
		/// <summary>Constructor</summary>
		/// <remarks>Values beginning or ending with 
		/// whitespace, or containing commas or semicolons, will automatically set 
		/// <see cref="IsQuoted"/> to true.</remarks>
		/// <param name="Value">Initial value.</param>
		/// <param name="IsQuoted">Specifies quotation of a value, see 
		/// <see cref="QUOTE_DELIMITER"/>.</param>
		public CIniValueItem(string Value, bool IsQuoted)
		{
			try
			{
				this._IsQuoted = IsQuoted;
				// Value has to be assigned after IsQuoted to allow subsequent adjustment 
				// of IsQuoted.
				this.Value = Value;
			}
			catch { throw; }
		}

		/// <summary>constructor</summary>
		/// <remarks>>Values beginning or ending with 
		/// whitespace, or containing commas or semicolons, will automatically set 
		/// <see cref="IsQuoted"/> to true.</remarks>
		/// <param name="Value">Initial value.</param>
		public CIniValueItem(string Value)
		{
			this.Value = Value;
		}
		#endregion

		#region constants
		/// <summary>Character to enclose string values beginning or ending with 
		/// whitespace, or containing commas or semicolons.</summary>
		public const char QUOTE_DELIMITER = '"';
		/// <summary><see cref="QUOTE_DELIMITER"/> characters appearing inside of
		/// string values enclosed in <see cref="QUOTE_DELIMITER"/> must be encoded 
		/// as escape sequences, containing this character followed by <see 
		/// cref="QUOTE_DELIMITER"/>.</summary>
		public const char ESCAPE = '\\';
		/// <summary>Characters inside of a string value enforcing string values
		/// to be enclosed in <see cref="QUOTE_DELIMITER"/>.</summary>
		public static readonly char[] COMPEL_QUOTE_DELIMITER = new char[] {',', ';'};
		/// <summary>Definition of whitespace: space, horizontal tab, line feed, 
		/// carriage return, vertical tab and form feed. 
		/// This whitespace characters have to match with characters deleted 
		/// by the framework method Trim()!
		/// </summary>
		public const string WHITESPACE_STRING = " \t\n\r\x000b\x000c";

		private const string MESSAGE_VALUE_STARTS_WITH_QUOTE = 
			"Value starts with a not escaped double quotation mark.";
		private const string MESSAGE_UNESCAPED_QUOTES = 
			"Not escaped double quotation marks inside of a quoted value.";
		#endregion

		#region members and properties
		/// <summary>See <see cref="IsQuoted"/>.</summary>
		protected bool _IsQuoted = false;
		/// <value>String values beginning or ending with whitespace (see
		/// <see cref="WHITESPACE_STRING"/>), or containing 
		/// commas or semicolons, must be enclosed in <see 
		/// cref="QUOTE_DELIMITER"/>.</value>
		public bool IsQuoted
		{
			get
			{
				return _IsQuoted;
			}
		}

		/// <summary>See <see cref="Value"/>.</summary>
		protected string _Value = null;
		/// <value><para><see cref="QUOTE_DELIMITER"/> characters appearing inside
		/// of string values enclosed in <see cref="QUOTE_DELIMITER"/> must be 
		/// encoded as escape sequences, containing <see cref="ESCAPE"/> followed 
		/// by <see cref="QUOTE_DELIMITER"/>. For string values not enclosed in 
		/// <see cref="QUOTE_DELIMITER"/> this is only necessary if beginning
		/// with a <see cref="QUOTE_DELIMITER"/> character.</para>
		/// The following two values are virtually the same:
		/// <code>
		/// "\"example value\""
		/// \"example value"
		/// </code> 
		/// <para><see cref="QUOTE_DELIMITER"/> inside of string values will not 
		/// be escaped or unescaped automatically!</para>
		/// <para>Null values will be changed to String.Empty.</para></value>
		/// <exception cref="System.Exception">Value starts with a not escaped 
		/// double quotation mark, or a quoted value contains not escaped double 
		/// quotation marks.</exception>
		public string Value
		{
			get
			{
				return _Value;
			}
			set
			{
				try
				{
					if (value == null)
					{
						_Value = String.Empty;
					}
					else
					{
						_Value = value;
					}

					// Check if Value must be enclosed in quotes.
					if (!IsQuoted)
					{
						// Trim the same characters as in WHITESPACE_STRING!
						string TempValue = _Value.Trim();
						if (_Value != TempValue || 
							  _Value.IndexOfAny(COMPEL_QUOTE_DELIMITER) > -1)
						{
							_IsQuoted = true;
						}
					} // if

					// Values beginning with a not escaped double quotation mark
					// rise an exception.
					if (_Value.Length > 0 && 
						  _Value.StartsWith(Convert.ToString(QUOTE_DELIMITER)))
					{
						throw new System.Exception(MESSAGE_VALUE_STARTS_WITH_QUOTE);
					}

					// Not escaped double quotation marks inside of a quoted value
					// rise an exception.
					if (IsQuoted && _Value.Length > 0)
					{
						int StartIndex = 0;

						StartIndex = _Value.IndexOf(QUOTE_DELIMITER);
						while (StartIndex > -1)
						{
							if (_Value[StartIndex - 1] != ESCAPE) // Checked range already.
							{
								throw new System.Exception(MESSAGE_UNESCAPED_QUOTES);
							}
							if (StartIndex + 1 >= _Value.Length) { break; }
							StartIndex = _Value.IndexOf(QUOTE_DELIMITER, StartIndex + 1);
						} // while
					} // if
				}
				catch { throw; }
			} // set
		}
		#endregion
	} // class

} // namespace
